/*
  Definition of constant tables
  This module contains most of the constant data that can be stored in ROM.
*/
#include <float.h>
#include <stdlib.h>
#include <stdio.h>
#include "FloatFR.h"
#include "sbrdecoder.h"
#include "sbr_rom.h"

const unsigned char sbr_start_freq_16[16] = {16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31};
const unsigned char sbr_start_freq_22[16] = {12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 26, 28, 30};
const unsigned char sbr_start_freq_24[16] = {11, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 25, 27, 29, 32};
const unsigned char sbr_start_freq_32[16] = {10, 12, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 25, 27, 29, 32};
const unsigned char sbr_start_freq_44[16] = { 8, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 21, 23, 25, 28, 32};
const unsigned char sbr_start_freq_48[16] = { 7,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 20, 22, 24, 27, 31};


/* Constants and tables used for envelope adjustment */

/* Mantissas of gain limits */
const float sbr_limGains[4] =
{
  0.5011932025f,
  1.0f,
  1.9952692516f,
  1.0e20f
};

/* Constants for calculating the number of limiter bands */
const float sbr_limiterBandsPerOctave[4] =
{
  1.0f,
  1.2f,
  2.0f,
  3.0f
};

/* Ratio of old gains and noise levels for the first 4 timeslots of an envelope */
const float sbr_smoothFilter[4] = {
  0.66666666666666f,
  0.36516383427084f,
  0.14699433520835f,
  0.03183050093751f
};


/* Lookup-table for inverse of integers */
const float sbr_invIntTable[INV_INT_TABLE_SIZE] = {
  FLT_MAX,    /* 1/0  should be inf */
  1.0f,
  1.0f / 2.0f,
  1.0f / 3.0f,
  1.0f / 4.0f,
  1.0f / 5.0f,
  1.0f / 6.0f,
  1.0f / 7.0f,
  1.0f / 8.0f,
  1.0f / 9.0f,
  1.0f / 10.0f,
  1.0f / 11.0f,
  1.0f / 12.0f,
  1.0f / 13.0f,
  1.0f / 14.0f,
  1.0f / 15.0f,
  1.0f / 16.0f,
  1.0f / 17.0f,
  1.0f / 18.0f,
  1.0f / 19.0f,
  1.0f / 20.0f,
  1.0f / 21.0f,
  1.0f / 22.0f,
  1.0f / 23.0f,
  1.0f / 24.0f,
  1.0f / 25.0f,
  1.0f / 26.0f,
  1.0f / 27.0f,
  1.0f / 28.0f,
  1.0f / 29.0f,
  1.0f / 30.0f,
  1.0f / 31.0f,
  1.0f / 32.0f,
  1.0f / 33.0f,
  1.0f / 34.0f,
  1.0f / 35.0f,
  1.0f / 36.0f,
  1.0f / 37.0f,
  1.0f / 38.0f,
  1.0f / 39.0f,
  1.0f / 40.0f,
  1.0f / 41.0f,
  1.0f / 42.0f,
  1.0f / 43.0f,
  1.0f / 44.0f,
  1.0f / 45.0f,
  1.0f / 46.0f,
  1.0f / 47.0f,
  1.0f / 48.0f,
  1.0f / 49.0f,
  1.0f / 50.0f,
  1.0f / 51.0f,
  1.0f / 52.0f,
  1.0f / 53.0f,
  1.0f / 54.0f
};


/* Real and imaginary part of random noise which will be modulated
  to the desired level. An accuracy of 12 bits is sufficient for these
  random numbers.
*/
const float sbr_randomPhase[SBR_NF_NO_RANDOM_VAL][2] = {
  {-0.99948153278296f, -0.59483417516607f},
  {0.97113454393991f, -0.67528515225647f},
  {0.14130051758487f, -0.95090983575689f},
  {-0.47005496701697f, -0.37340549728647f},
  {0.80705063769351f, 0.29653668284408f},
  {-0.38981478896926f, 0.89572605717087f},
  {-0.01053049862020f, -0.66959058036166f},
  {-0.91266367957293f, -0.11522938140034f},
  {0.54840422910309f, 0.75221367176302f},
  {0.40009252867955f, -0.98929400334421f},
  {-0.99867974711855f, -0.88147068645358f},
  {-0.95531076805040f, 0.90908757154593f},
  {-0.45725933317144f, -0.56716323646760f},
  {-0.72929675029275f, -0.98008272727324f},
  {0.75622801399036f, 0.20950329995549f},
  {0.07069442601050f, -0.78247898470706f},
  {0.74496252926055f, -0.91169004445807f},
  {-0.96440182703856f, -0.94739918296622f},
  {0.30424629369539f, -0.49438267012479f},
  {0.66565033746925f, 0.64652935542491f},
  {0.91697008020594f, 0.17514097332009f},
  {-0.70774918760427f, 0.52548653416543f},
  {-0.70051415345560f, -0.45340028808763f},
  {-0.99496513054797f, -0.90071908066973f},
  {0.98164490790123f, -0.77463155528697f},
  {-0.54671580548181f, -0.02570928536004f},
  {-0.01689629065389f, 0.00287506445732f},
  {-0.86110349531986f, 0.42548583726477f},
  {-0.98892980586032f, -0.87881132267556f},
  {0.51756627678691f, 0.66926784710139f},
  {-0.99635026409640f, -0.58107730574765f},
  {-0.99969370862163f, 0.98369989360250f},
  {0.55266258627194f, 0.59449057465591f},
  {0.34581177741673f, 0.94879421061866f},
  {0.62664209577999f, -0.74402970906471f},
  {-0.77149701404973f, -0.33883658042801f},
  {-0.91592244254432f, 0.03687901376713f},
  {-0.76285492357887f, -0.91371867919124f},
  {0.79788337195331f, -0.93180971199849f},
  {0.54473080610200f, -0.11919206037186f},
  {-0.85639281671058f, 0.42429854760451f},
  {-0.92882402971423f, 0.27871809078609f},
  {-0.11708371046774f, -0.99800843444966f},
  {0.21356749817493f, -0.90716295627033f},
  {-0.76191692573909f, 0.99768118356265f},
  {0.98111043100884f, -0.95854459734407f},
  {-0.85913269895572f, 0.95766566168880f},
  {-0.93307242253692f, 0.49431757696466f},
  {0.30485754879632f, -0.70540034357529f},
  {0.85289650925190f, 0.46766131791044f},
  {0.91328082618125f, -0.99839597361769f},
  {-0.05890199924154f, 0.70741827819497f},
  {0.28398686150148f, 0.34633555702188f},
  {0.95258164539612f, -0.54893416026939f},
  {-0.78566324168507f, -0.75568541079691f},
  {-0.95789495447877f, -0.20423194696966f},
  {0.82411158711197f, 0.96654618432562f},
  {-0.65185446735885f, -0.88734990773289f},
  {-0.93643603134666f, 0.99870790442385f},
  {0.91427159529618f, -0.98290505544444f},
  {-0.70395684036886f, 0.58796798221039f},
  {0.00563771969365f, 0.61768196727244f},
  {0.89065051931895f, 0.52783352697585f},
  {-0.68683707712762f, 0.80806944710339f},
  {0.72165342518718f, -0.69259857349564f},
  {-0.62928247730667f, 0.13627037407335f},
  {0.29938434065514f, -0.46051329682246f},
  {-0.91781958879280f, -0.74012716684186f},
  {0.99298717043688f, 0.40816610075661f},
  {0.82368298622748f, -0.74036047190173f},
  {-0.98512833386833f, -0.99972330709594f},
  {-0.95915368242257f, -0.99237800466040f},
  {-0.21411126572790f, -0.93424819052545f},
  {-0.68821476106884f, -0.26892306315457f},
  {0.91851997982317f, 0.09358228901785f},
  {-0.96062769559127f, 0.36099095133739f},
  {0.51646184922287f, -0.71373332873917f},
  {0.61130721139669f, 0.46950141175917f},
  {0.47336129371299f, -0.27333178296162f},
  {0.90998308703519f, 0.96715662938132f},
  {0.44844799194357f, 0.99211574628306f},
  {0.66614891079092f, 0.96590176169121f},
  {0.74922239129237f, -0.89879858826087f},
  {-0.99571588506485f, 0.52785521494349f},
  {0.97401082477563f, -0.16855870075190f},
  {0.72683747733879f, -0.48060774432251f},
  {0.95432193457128f, 0.68849603408441f},
  {-0.72962208425191f, -0.76608443420917f},
  {-0.85359479233537f, 0.88738125901579f},
  {-0.81412430338535f, -0.97480768049637f},
  {-0.87930772356786f, 0.74748307690436f},
  {-0.71573331064977f, -0.98570608178923f},
  {0.83524300028228f, 0.83702537075163f},
  {-0.48086065601423f, -0.98848504923531f},
  {0.97139128574778f, 0.80093621198236f},
  {0.51992825347895f, 0.80247631400510f},
  {-0.00848591195325f, -0.76670128000486f},
  {-0.70294374303036f, 0.55359910445577f},
  {-0.95894428168140f, -0.43265504344783f},
  {0.97079252950321f, 0.09325857238682f},
  {-0.92404293670797f, 0.85507704027855f},
  {-0.69506469500450f, 0.98633412625459f},
  {0.26559203620024f, 0.73314307966524f},
  {0.28038443336943f, 0.14537913654427f},
  {-0.74138124825523f, 0.99310339807762f},
  {-0.01752795995444f, -0.82616635284178f},
  {-0.55126773094930f, -0.98898543862153f},
  {0.97960898850996f, -0.94021446752851f},
  {-0.99196309146936f, 0.67019017358456f},
  {-0.67684928085260f, 0.12631491649378f},
  {0.09140039465500f, -0.20537731453108f},
  {-0.71658965751996f, -0.97788200391224f},
  {0.81014640078925f, 0.53722648362443f},
  {0.40616991671205f, -0.26469008598449f},
  {-0.67680188682972f, 0.94502052337695f},
  {0.86849774348749f, -0.18333598647899f},
  {-0.99500381284851f, -0.02634122068550f},
  {0.84329189340667f, 0.10406957462213f},
  {-0.09215968531446f, 0.69540012101253f},
  {0.99956173327206f, -0.12358542001404f},
  {-0.79732779473535f, -0.91582524736159f},
  {0.96349973642406f, 0.96640458041000f},
  {-0.79942778496547f, 0.64323902822857f},
  {-0.11566039853896f, 0.28587846253726f},
  {-0.39922954514662f, 0.94129601616966f},
  {0.99089197565987f, -0.92062625581587f},
  {0.28631285179909f, -0.91035047143603f},
  {-0.83302725605608f, -0.67330410892084f},
  {0.95404443402072f, 0.49162765398743f},
  {-0.06449863579434f, 0.03250560813135f},
  {-0.99575054486311f, 0.42389784469507f},
  {-0.65501142790847f, 0.82546114655624f},
  {-0.81254441908887f, -0.51627234660629f},
  {-0.99646369485481f, 0.84490533520752f},
  {0.00287840603348f, 0.64768261158166f},
  {0.70176989408455f, -0.20453028573322f},
  {0.96361882270190f, 0.40706967140989f},
  {-0.68883758192426f, 0.91338958840772f},
  {-0.34875585502238f, 0.71472290693300f},
  {0.91980081243087f, 0.66507455644919f},
  {-0.99009048343881f, 0.85868021604848f},
  {0.68865791458395f, 0.55660316809678f},
  {-0.99484402129368f, -0.20052559254934f},
  {0.94214511408023f, -0.99696425367461f},
  {-0.67414626793544f, 0.49548221180078f},
  {-0.47339353684664f, -0.85904328834047f},
  {0.14323651387360f, -0.94145598222488f},
  {-0.29268293575672f, 0.05759224927952f},
  {0.43793861458754f, -0.78904969892724f},
  {-0.36345126374441f, 0.64874435357162f},
  {-0.08750604656825f, 0.97686944362527f},
  {-0.96495267812511f, -0.53960305946511f},
  {0.55526940659947f, 0.78891523734774f},
  {0.73538215752630f, 0.96452072373404f},
  {-0.30889773919437f, -0.80664389776860f},
  {0.03574995626194f, -0.97325616900959f},
  {0.98720684660488f, 0.48409133691962f},
  {-0.81689296271203f, -0.90827703628298f},
  {0.67866860118215f, 0.81284503870856f},
  {-0.15808569732583f, 0.85279555024382f},
  {0.80723395114371f, -0.24717418514605f},
  {0.47788757329038f, -0.46333147839295f},
  {0.96367554763201f, 0.38486749303242f},
  {-0.99143875716818f, -0.24945277239809f},
  {0.83081876925833f, -0.94780851414763f},
  {-0.58753191905341f, 0.01290772389163f},
  {0.95538108220960f, -0.85557052096538f},
  {-0.96490920476211f, -0.64020970923102f},
  {-0.97327101028521f, 0.12378128133110f},
  {0.91400366022124f, 0.57972471346930f},
  {-0.99925837363824f, 0.71084847864067f},
  {-0.86875903507313f, -0.20291699203564f},
  {-0.26240034795124f, -0.68264554369108f},
  {-0.24664412953388f, -0.87642273115183f},
  {0.02416275806869f, 0.27192914288905f},
  {0.82068619590515f, -0.85087787994476f},
  {0.88547373760759f, -0.89636802901469f},
  {-0.18173078152226f, -0.26152145156800f},
  {0.09355476558534f, 0.54845123045604f},
  {-0.54668414224090f, 0.95980774020221f},
  {0.37050990604091f, -0.59910140383171f},
  {-0.70373594262891f, 0.91227665827081f},
  {-0.34600785879594f, -0.99441426144200f},
  {-0.68774481731008f, -0.30238837956299f},
  {-0.26843291251234f, 0.83115668004362f},
  {0.49072334613242f, -0.45359708737775f},
  {0.38975993093975f, 0.95515358099121f},
  {-0.97757125224150f, 0.05305894580606f},
  {-0.17325552859616f, -0.92770672250494f},
  {0.99948035025744f, 0.58285545563426f},
  {-0.64946246527458f, 0.68645507104960f},
  {-0.12016920576437f, -0.57147322153312f},
  {-0.58947456517751f, -0.34847132454388f},
  {-0.41815140454465f, 0.16276422358861f},
  {0.99885650204884f, 0.11136095490444f},
  {-0.56649614128386f, -0.90494866361587f},
  {0.94138021032330f, 0.35281916733018f},
  {-0.75725076534641f, 0.53650549640587f},
  {0.20541973692630f, -0.94435144369918f},
  {0.99980371023351f, 0.79835913565599f},
  {0.29078277605775f, 0.35393777921520f},
  {-0.62858772103030f, 0.38765693387102f},
  {0.43440904467688f, -0.98546330463232f},
  {-0.98298583762390f, 0.21021524625209f},
  {0.19513029146934f, -0.94239832251867f},
  {-0.95476662400101f, 0.98364554179143f},
  {0.93379635304810f, -0.70881994583682f},
  {-0.85235410573336f, -0.08342347966410f},
  {-0.86425093011245f, -0.45795025029466f},
  {0.38879779059045f, 0.97274429344593f},
  {0.92045124735495f, -0.62433652524220f},
  {0.89162532251878f, 0.54950955570563f},
  {-0.36834336949252f, 0.96458298020975f},
  {0.93891760988045f, -0.89968353740388f},
  {0.99267657565094f, -0.03757034316958f},
  {-0.94063471614176f, 0.41332338538963f},
  {0.99740224117019f, -0.16830494996370f},
  {-0.35899413170555f, -0.46633226649613f},
  {0.05237237274947f, -0.25640361602661f},
  {0.36703583957424f, -0.38653265641875f},
  {0.91653180367913f, -0.30587628726597f},
  {0.69000803499316f, 0.90952171386132f},
  {-0.38658751133527f, 0.99501571208985f},
  {-0.29250814029851f, 0.37444994344615f},
  {-0.60182204677608f, 0.86779651036123f},
  {-0.97418588163217f, 0.96468523666475f},
  {0.88461574003963f, 0.57508405276414f},
  {0.05198933055162f, 0.21269661669964f},
  {-0.53499621979720f, 0.97241553731237f},
  {-0.49429560226497f, 0.98183865291903f},
  {-0.98935142339139f, -0.40249159006933f},
  {-0.98081380091130f, -0.72856895534041f},
  {-0.27338148835532f, 0.99950922447209f},
  {0.06310802338302f, -0.54539587529618f},
  {-0.20461677199539f, -0.14209977628489f},
  {0.66223843141647f, 0.72528579940326f},
  {-0.84764345483665f, 0.02372316801261f},
  {-0.89039863483811f, 0.88866581484602f},
  {0.95903308477986f, 0.76744927173873f},
  {0.73504123909879f, -0.03747203173192f},
  {-0.31744434966056f, -0.36834111883652f},
  {-0.34110827591623f, 0.40211222807691f},
  {0.47803883714199f, -0.39423219786288f},
  {0.98299195879514f, 0.01989791390047f},
  {-0.30963073129751f, -0.18076720599336f},
  {0.99992588229018f, -0.26281872094289f},
  {-0.93149731080767f, -0.98313162570490f},
  {0.99923472302773f, -0.80142993767554f},
  {-0.26024169633417f, -0.75999759855752f},
  {-0.35712514743563f, 0.19298963768574f},
  {-0.99899084509530f, 0.74645156992493f},
  {0.86557171579452f, 0.55593866696299f},
  {0.33408042438752f, 0.86185953874709f},
  {0.99010736374716f, 0.04602397576623f},
  {-0.66694269691195f, -0.91643611810148f},
  {0.64016792079480f, 0.15649530836856f},
  {0.99570534804836f, 0.45844586038111f},
  {-0.63431466947340f, 0.21079116459234f},
  {-0.07706847005931f, -0.89581437101329f},
  {0.98590090577724f, 0.88241721133981f},
  {0.80099335254678f, -0.36851896710853f},
  {0.78368131392666f, 0.45506999802597f},
  {0.08707806671691f, 0.80938994918745f},
  {-0.86811883080712f, 0.39347308654705f},
  {-0.39466529740375f, -0.66809432114456f},
  {0.97875325649683f, -0.72467840967746f},
  {-0.95038560288864f, 0.89563219587625f},
  {0.17005239424212f, 0.54683053962658f},
  {-0.76910792026848f, -0.96226617549298f},
  {0.99743281016846f, 0.42697157037567f},
  {0.95437383549973f, 0.97002324109952f},
  {0.99578905365569f, -0.54106826257356f},
  {0.28058259829990f, -0.85361420634036f},
  {0.85256524470573f, -0.64567607735589f},
  {-0.50608540105128f, -0.65846015480300f},
  {-0.97210735183243f, -0.23095213067791f},
  {0.95424048234441f, -0.99240147091219f},
  {-0.96926570524023f, 0.73775654896574f},
  {0.30872163214726f, 0.41514960556126f},
  {-0.24523839572639f, 0.63206633394807f},
  {-0.33813265086024f, -0.38661779441897f},
  {-0.05826828420146f, -0.06940774188029f},
  {-0.22898461455054f, 0.97054853316316f},
  {-0.18509915019881f, 0.47565762892084f},
  {-0.10488238045009f, -0.87769947402394f},
  {-0.71886586182037f, 0.78030982480538f},
  {0.99793873738654f, 0.90041310491497f},
  {0.57563307626120f, -0.91034337352097f},
  {0.28909646383717f, 0.96307783970534f},
  {0.42188998312520f, 0.48148651230437f},
  {0.93335049681047f, -0.43537023883588f},
  {-0.97087374418267f, 0.86636445711364f},
  {0.36722871286923f, 0.65291654172961f},
  {-0.81093025665696f, 0.08778370229363f},
  {-0.26240603062237f, -0.92774095379098f},
  {0.83996497984604f, 0.55839849139647f},
  {-0.99909615720225f, -0.96024605713970f},
  {0.74649464155061f, 0.12144893606462f},
  {-0.74774595569805f, -0.26898062008959f},
  {0.95781667469567f, -0.79047927052628f},
  {0.95472308713099f, -0.08588776019550f},
  {0.48708332746299f, 0.99999041579432f},
  {0.46332038247497f, 0.10964126185063f},
  {-0.76497004940162f, 0.89210929242238f},
  {0.57397389364339f, 0.35289703373760f},
  {0.75374316974495f, 0.96705214651335f},
  {-0.59174397685714f, -0.89405370422752f},
  {0.75087906691890f, -0.29612672982396f},
  {-0.98607857336230f, 0.25034911730023f},
  {-0.40761056640505f, -0.90045573444695f},
  {0.66929266740477f, 0.98629493401748f},
  {-0.97463695257310f, -0.00190223301301f},
  {0.90145509409859f, 0.99781390365446f},
  {-0.87259289048043f, 0.99233587353666f},
  {-0.91529461447692f, -0.15698707534206f},
  {-0.03305738840705f, -0.37205262859764f},
  {0.07223051368337f, -0.88805001733626f},
  {0.99498012188353f, 0.97094358113387f},
  {-0.74904939500519f, 0.99985483641521f},
  {0.04585228574211f, 0.99812337444082f},
  {-0.89054954257993f, -0.31791913188064f},
  {-0.83782144651251f, 0.97637632547466f},
  {0.33454804933804f, -0.86231516800408f},
  {-0.99707579362824f, 0.93237990079441f},
  {-0.22827527843994f, 0.18874759397997f},
  {0.67248046289143f, -0.03646211390569f},
  {-0.05146538187944f, -0.92599700120679f},
  {0.99947295749905f, 0.93625229707912f},
  {0.66951124390363f, 0.98905825623893f},
  {-0.99602956559179f, -0.44654715757688f},
  {0.82104905483590f, 0.99540741724928f},
  {0.99186510988782f, 0.72023001312947f},
  {-0.65284592392918f, 0.52186723253637f},
  {0.93885443798188f, -0.74895312615259f},
  {0.96735248738388f, 0.90891816978629f},
  {-0.22225968841114f, 0.57124029781228f},
  {-0.44132783753414f, -0.92688840659280f},
  {-0.85694974219574f, 0.88844532719844f},
  {0.91783042091762f, -0.46356892383970f},
  {0.72556974415690f, -0.99899555770747f},
  {-0.99711581834508f, 0.58211560180426f},
  {0.77638976371966f, 0.94321834873819f},
  {0.07717324253925f, 0.58638399856595f},
  {-0.56049829194163f, 0.82522301569036f},
  {0.98398893639988f, 0.39467440420569f},
  {0.47546946844938f, 0.68613044836811f},
  {0.65675089314631f, 0.18331637134880f},
  {0.03273375457980f, -0.74933109564108f},
  {-0.38684144784738f, 0.51337349030406f},
  {-0.97346267944545f, -0.96549364384098f},
  {-0.53282156061942f, -0.91423265091354f},
  {0.99817310731176f, 0.61133572482148f},
  {-0.50254500772635f, -0.88829338134294f},
  {0.01995873238855f, 0.85223515096765f},
  {0.99930381973804f, 0.94578896296649f},
  {0.82907767600783f, -0.06323442598128f},
  {-0.58660709669728f, 0.96840773806582f},
  {-0.17573736667267f, -0.48166920859485f},
  {0.83434292401346f, -0.13023450646997f},
  {0.05946491307025f, 0.20511047074866f},
  {0.81505484574602f, -0.94685947861369f},
  {-0.44976380954860f, 0.40894572671545f},
  {-0.89746474625671f, 0.99846578838537f},
  {0.39677256130792f, -0.74854668609359f},
  {-0.07588948563079f, 0.74096214084170f},
  {0.76343198951445f, 0.41746629422634f},
  {-0.74490104699626f, 0.94725911744610f},
  {0.64880119792759f, 0.41336660830571f},
  {0.62319537462542f, -0.93098313552599f},
  {0.42215817594807f, -0.07712787385208f},
  {0.02704554141885f, -0.05417518053666f},
  {0.80001773566818f, 0.91542195141039f},
  {-0.79351832348816f, -0.36208897989136f},
  {0.63872359151636f, 0.08128252493444f},
  {0.52890520960295f, 0.60048872455592f},
  {0.74238552914587f, 0.04491915291044f},
  {0.99096131449250f, -0.19451182854402f},
  {-0.80412329643109f, -0.88513818199457f},
  {-0.64612616129736f, 0.72198674804544f},
  {0.11657770663191f, -0.83662833815041f},
  {-0.95053182488101f, -0.96939905138082f},
  {-0.62228872928622f, 0.82767262846661f},
  {0.03004475787316f, -0.99738896333384f},
  {-0.97987214341034f, 0.36526129686425f},
  {-0.99986980746200f, -0.36021610299715f},
  {0.89110648599879f, -0.97894250343044f},
  {0.10407960510582f, 0.77357793811619f},
  {0.95964737821728f, -0.35435818285502f},
  {0.50843233159162f, 0.96107691266205f},
  {0.17006334670615f, -0.76854025314829f},
  {0.25872675063360f, 0.99893303933816f},
  {-0.01115998681937f, 0.98496019742444f},
  {-0.79598702973261f, 0.97138411318894f},
  {-0.99264708948101f, -0.99542822402536f},
  {-0.99829663752818f, 0.01877138824311f},
  {-0.70801016548184f, 0.33680685948117f},
  {-0.70467057786826f, 0.93272777501857f},
  {0.99846021905254f, -0.98725746254433f},
  {-0.63364968534650f, -0.16473594423746f},
  {-0.16258217500792f, -0.95939125400802f},
  {-0.43645594360633f, -0.94805030113284f},
  {-0.99848471702976f, 0.96245166923809f},
  {-0.16796458968998f, -0.98987511890470f},
  {-0.87979225745213f, -0.71725725041680f},
  {0.44183099021786f, -0.93568974498761f},
  {0.93310180125532f, -0.99913308068246f},
  {-0.93941931782002f, -0.56409379640356f},
  {-0.88590003188677f, 0.47624600491382f},
  {0.99971463703691f, -0.83889954253462f},
  {-0.75376385639978f, 0.00814643438625f},
  {0.93887685615875f, -0.11284528204636f},
  {0.85126435782309f, 0.52349251543547f},
  {0.39701421446381f, 0.81779634174316f},
  {-0.37024464187437f, -0.87071656222959f},
  {-0.36024828242896f, 0.34655735648287f},
  {-0.93388812549209f, -0.84476541096429f},
  {-0.65298804552119f, -0.18439575450921f},
  {0.11960319006843f, 0.99899346780168f},
  {0.94292565553160f, 0.83163906518293f},
  {0.75081145286948f, -0.35533223142265f},
  {0.56721979748394f, -0.24076836414499f},
  {0.46857766746029f, -0.30140233457198f},
  {0.97312313923635f, -0.99548191630031f},
  {-0.38299976567017f, 0.98516909715427f},
  {0.41025800019463f, 0.02116736935734f},
  {0.09638062008048f, 0.04411984381457f},
  {-0.85283249275397f, 0.91475563922421f},
  {0.88866808958124f, -0.99735267083226f},
  {-0.48202429536989f, -0.96805608884164f},
  {0.27572582416567f, 0.58634753335832f},
  {-0.65889129659168f, 0.58835634138583f},
  {0.98838086953732f, 0.99994349600236f},
  {-0.20651349620689f, 0.54593044066355f},
  {-0.62126416356920f, -0.59893681700392f},
  {0.20320105410437f, -0.86879180355289f},
  {-0.97790548600584f, 0.96290806999242f},
  {0.11112534735126f, 0.21484763313301f},
  {-0.41368337314182f, 0.28216837680365f},
  {0.24133038992960f, 0.51294362630238f},
  {-0.66393410674885f, -0.08249679629081f},
  {-0.53697829178752f, -0.97649903936228f},
  {-0.97224737889348f, 0.22081333579837f},
  {0.87392477144549f, -0.12796173740361f},
  {0.19050361015753f, 0.01602615387195f},
  {-0.46353441212724f, -0.95249041539006f},
  {-0.07064096339021f, -0.94479803205886f},
  {-0.92444085484466f, -0.10457590187436f},
  {-0.83822593578728f, -0.01695043208885f},
  {0.75214681811150f, -0.99955681042665f},
  {-0.42102998829339f, 0.99720941999394f},
  {-0.72094786237696f, -0.35008961934255f},
  {0.78843311019251f, 0.52851398958271f},
  {0.97394027897442f, -0.26695944086561f},
  {0.99206463477946f, -0.57010120849429f},
  {0.76789609461795f, -0.76519356730966f},
  {-0.82002421836409f, -0.73530179553767f},
  {0.81924990025724f, 0.99698425250579f},
  {-0.26719850873357f, 0.68903369776193f},
  {-0.43311260380975f, 0.85321815947490f},
  {0.99194979673836f, 0.91876249766422f},
  {-0.80692001248487f, -0.32627540663214f},
  {0.43080003649976f, -0.21919095636638f},
  {0.67709491937357f, -0.95478075822906f},
  {0.56151770568316f, -0.70693811747778f},
  {0.10831862810749f, -0.08628837174592f},
  {0.91229417540436f, -0.65987351408410f},
  {-0.48972893932274f, 0.56289246362686f},
  {-0.89033658689697f, -0.71656563987082f},
  {0.65269447475094f, 0.65916004833932f},
  {0.67439478141121f, -0.81684380846796f},
  {-0.47770832416973f, -0.16789556203025f},
  {-0.99715979260878f, -0.93565784007648f},
  {-0.90889593602546f, 0.62034397054380f},
  {-0.06618622548177f, -0.23812217221359f},
  {0.99430266919728f, 0.18812555317553f},
  {0.97686402381843f, -0.28664534366620f},
  {0.94813650221268f, -0.97506640027128f},
  {-0.95434497492853f, -0.79607978501983f},
  {-0.49104783137150f, 0.32895214359663f},
  {0.99881175120751f, 0.88993983831354f},
  {0.50449166760303f, -0.85995072408434f},
  {0.47162891065108f, -0.18680204049569f},
  {-0.62081581361840f, 0.75000676218956f},
  {-0.43867015250812f, 0.99998069244322f},
  {0.98630563232075f, -0.53578899600662f},
  {-0.61510362277374f, -0.89515019899997f},
  {-0.03841517601843f, -0.69888815681179f},
  {-0.30102157304644f, -0.07667808922205f},
  {0.41881284182683f, 0.02188098922282f},
  {-0.86135454941237f, 0.98947480909359f},
  {0.67226861393788f, -0.13494389011014f},
  {-0.70737398842068f, -0.76547349325992f},
  {0.94044946687963f, 0.09026201157416f},
  {-0.82386352534327f, 0.08924768823676f},
  {-0.32070666698656f, 0.50143421908753f},
  {0.57593163224487f, -0.98966422921509f},
  {-0.36326018419965f, 0.07440243123228f},
  {0.99979044674350f, -0.14130287347405f},
  {-0.92366023326932f, -0.97979298068180f},
  {-0.44607178518598f, -0.54233252016394f},
  {0.44226800932956f, 0.71326756742752f},
  {0.03671907158312f, 0.63606389366675f},
  {0.52175424682195f, -0.85396826735705f},
  {-0.94701139690956f, -0.01826348194255f},
  {-0.98759606946049f, 0.82288714303073f},
  {0.87434794743625f, 0.89399495655433f},
  {-0.93412041758744f, 0.41374052024363f},
  {0.96063943315511f, 0.93116709541280f},
  {0.97534253457837f, 0.86150930812689f},
  {0.99642466504163f, 0.70190043427512f},
  {-0.94705089665984f, -0.29580042814306f},
  {0.91599807087376f, -0.98147830385781f}
};
//@}


/*
  \name 	QMF
  \brief  QMF-Table
          64 channels, N = 640, optimized by PE 010516

  Symmetric properties of qmf coeffs:
  Use point symmetry:

  sbr_qmf_64_640_qmf[320..634] = p_64_640_qmf[314..0]

  \showinitializer
*/
//@{

const float sbr_qmf_64_640[325] =
{
-5.525286504714797e-004f,+1.439046660791857e-002f,+3.723795546306128e-001f,-3.499914122930965e-001f,-1.218499959507776e-002f,
-5.617692573788820e-004f,+1.554055533423407e-002f,+3.836350013904333e-001f,-3.388722693866466e-001f,-1.113155480320807e-002f,
-4.947518089560348e-004f,+1.673247129988964e-002f,+3.949211761567527e-001f,-3.278113727210475e-001f,-1.011502154985735e-002f,
-4.875227971213061e-004f,+1.794333813443166e-002f,+4.062317676762485e-001f,-3.168278913645616e-001f,-9.132532960846242e-003f,
-4.893791249849297e-004f,+1.918724313697539e-002f,+4.175696896840907e-001f,-3.059098575191602e-001f,-8.179823337263956e-003f,
-5.040714349717598e-004f,+2.045317933555374e-002f,+4.289119920737320e-001f,-2.950716717064639e-001f,-7.261581685172976e-003f,
-5.226564297171536e-004f,+2.174675502535081e-002f,+4.402553754366491e-001f,-2.843214189108535e-001f,-6.379229326851348e-003f,
-5.466565633704553e-004f,+2.306801692862206e-002f,+4.515996535682373e-001f,-2.736634040562456e-001f,-5.533721110879950e-003f,
-5.677802561282153e-004f,+2.441609920284815e-002f,+4.629308085275695e-001f,-2.631053299460340e-001f,-4.722259624000638e-003f,
-5.870930485225149e-004f,+2.578758475467104e-002f,+4.742453214611543e-001f,-2.526480309572177e-001f,-3.940112405218538e-003f,
-6.132747393831305e-004f,+2.718594296328715e-002f,+4.855253091109857e-001f,-2.423016884597387e-001f,-3.193377839003896e-003f,
-6.312493531935796e-004f,+2.860721736384611e-002f,+4.967708254570692e-001f,-2.320690870679074e-001f,-2.482672364493203e-003f,
-6.540333362057484e-004f,+3.005026574279500e-002f,+5.079817500043367e-001f,-2.219652696414943e-001f,-1.803947258930832e-003f,
-6.777690776369781e-004f,+3.150176087389379e-002f,+5.191234970239136e-001f,-2.119735853805564e-001f,-1.156813552266513e-003f,
-6.941614627303249e-004f,+3.297540810337295e-002f,+5.302240895685498e-001f,-2.021250176810344e-001f,-5.464280866436426e-004f,
-7.157736474351309e-004f,+3.446209487686384e-002f,+5.412553448732201e-001f,-1.923966745726700e-001f,+2.760451905299060e-005f,
-7.255043122205504e-004f,+3.596975605541700e-002f,+5.522051258506063e-001f,-1.828172548514229e-001f,+5.832264248006948e-004f,
-7.440941854094920e-004f,+3.748128504251629e-002f,+5.630789140137003e-001f,-1.733808172170604e-001f,+1.090232905120221e-003f,
-7.490598053217391e-004f,+3.900536794745497e-002f,+5.738524131692298e-001f,-1.640958855666853e-001f,+1.578468257676472e-003f,
-7.681371927024808e-004f,+4.053491705584002e-002f,+5.845403235467916e-001f,-1.549607071060459e-001f,+2.027417618500581e-003f,
-7.724848594888575e-004f,+4.206490946366589e-002f,+5.951123086249566e-001f,-1.459766491186991e-001f,+2.450854003212794e-003f,
-7.834332287666293e-004f,+4.360975421303862e-002f,+6.055783538918024e-001f,-1.371551761193432e-001f,+2.844675786234913e-003f,
-7.779869492680295e-004f,+4.514884056413160e-002f,+6.159109932029110e-001f,-1.285002850387848e-001f,+3.209188580984976e-003f,
-7.803664709970310e-004f,+4.668430272642297e-002f,+6.261242695605493e-001f,-1.200077984679968e-001f,+3.540124655067661e-003f,
-7.801449625711068e-004f,+4.821657200671922e-002f,+6.361980107728613e-001f,-1.116826931773024e-001f,+3.845640812460432e-003f,
-7.757977330969382e-004f,+4.973857556013930e-002f,+6.461269695946109e-001f,-1.035329531146302e-001f,+4.125164232698537e-003f,
-7.630793575703050e-004f,+5.125561555215880e-002f,+6.559016302467067e-001f,-9.555333528914223e-002f,+4.380186174471733e-003f,
-7.530001420101251e-004f,+5.276307465206772e-002f,+6.655139880162695e-001f,-8.775475365592653e-002f,+4.603953014705050e-003f,
-7.319357152453665e-004f,+5.424527683588578e-002f,+6.749663190171162e-001f,-8.013729344279387e-002f,+4.810946905999106e-003f,
-7.215391987622286e-004f,+5.571736482138088e-002f,+6.842353293459781e-001f,-7.269433008129310e-002f,+4.983968776294583e-003f,
-6.917937537191035e-004f,+5.716164501298929e-002f,+6.933282376703176e-001f,-6.544098531359448e-002f,+5.138227545142836e-003f,
-6.650415089345047e-004f,+5.859156836259560e-002f,+7.022388719353935e-001f,-5.837053268336145e-002f,+5.271575872723485e-003f,
-6.341594902471489e-004f,+5.998374801761285e-002f,+7.109410426309526e-001f,-5.148041767933625e-002f,+5.383897589704628e-003f,
-5.946118933031955e-004f,+6.134551717206593e-002f,+7.194462634956096e-001f,-4.478068215855852e-002f,+5.475378307696386e-003f,
-5.564576390631206e-004f,+6.268578081171672e-002f,+7.277448900299428e-001f,-3.827765720821592e-002f,+5.540436394000088e-003f,
-5.145572210825134e-004f,+6.397158980680977e-002f,+7.358211758276898e-001f,-3.195312745331961e-002f,+5.591712866303156e-003f,
-4.606325480338262e-004f,+6.522471064379697e-002f,+7.436827863648809e-001f,-2.582272888063778e-002f,+5.626611419323720e-003f,
-4.095121452206994e-004f,+6.643675122104309e-002f,+7.513137456123731e-001f,-1.988341292573269e-002f,+5.638919951508487e-003f,
-3.501175875595919e-004f,+6.760759851227793e-002f,+7.587080760824195e-001f,-1.412888273557668e-002f,+5.645519691635520e-003f,
-2.896981174825215e-004f,+6.870438283511596e-002f,+7.658674865093896e-001f,-8.571174913661668e-003f,+5.622064320965902e-003f,
-2.098337343963717e-004f,+6.976302447126595e-002f,+7.727780881332725e-001f,-3.208689683036121e-003f,+5.593802300454330e-003f,
-1.446380934949981e-004f,+7.076287107265650e-002f,+7.794287519021588e-001f,+1.976560145028481e-003f,+5.547571450883520e-003f,
-6.173344071970479e-005f,+7.170026731101810e-002f,+7.858353120392010e-001f,+6.963686216167498e-003f,+5.487604015074597e-003f,
+1.349497417596386e-005f,+7.256825833083160e-002f,+7.919735841642436e-001f,+1.176238327857339e-002f,+5.419677593070989e-003f,
+1.094383127384685e-004f,+7.336202550803318e-002f,+7.978466413770040e-001f,+1.637012582228135e-002f,+5.347168119820721e-003f,
+2.043017068780772e-004f,+7.410036424341697e-002f,+8.034485751850518e-001f,+2.079970728622271e-002f,+5.246116613236508e-003f,
+2.949531104116364e-004f,+7.474525581193675e-002f,+8.087695004449111e-001f,+2.503075618908813e-002f,+5.140735390322959e-003f,
+4.026540215999252e-004f,+7.531373362019285e-002f,+8.138191270621699e-001f,+2.908240060124874e-002f,+5.039302260125202e-003f,
+5.107388495248741e-004f,+7.580083586584310e-002f,+8.185776004646774e-001f,+3.295839306690847e-002f,+4.913760357448478e-003f,
+6.239376139142985e-004f,+7.619924793395567e-002f,+8.230419890540879e-001f,+3.664181168133450e-002f,+4.793256084983462e-003f,
+7.458025886538838e-004f,+7.649921704119105e-002f,+8.272275347335977e-001f,+4.014582784127306e-002f,+4.660646061176193e-003f,
+8.608443326193891e-004f,+7.670934904244611e-002f,+8.311038457151991e-001f,+4.347687821958321e-002f,+4.520985278247952e-003f,
+9.885988301536750e-004f,+7.681739756964497e-002f,+8.346937361840173e-001f,+4.663033051701176e-002f,+4.373071967813425e-003f,
+1.125015513066064e-003f,+7.682300113922813e-002f,+8.379717337886474e-001f,+4.959786763444761e-002f,+4.226426922703637e-003f,
+1.257788464745049e-003f,+7.672049241745922e-002f,+8.409541392472186e-001f,+5.240938217365652e-002f,+4.081975319350373e-003f,
+1.390249482718497e-003f,+7.650507183193828e-002f,+8.436238281200542e-001f,+5.504600343008619e-002f,+3.920743237034152e-003f,
+1.544321984713545e-003f,+7.617483218536458e-002f,+8.459818469820640e-001f,+5.751526919867195e-002f,+3.760392291041691e-003f,
+1.686808325309625e-003f,+7.573057565061056e-002f,+8.480315777076319e-001f,+5.981665708090339e-002f,+3.600826812313299e-003f,
+1.834826542238209e-003f,+7.515762552869984e-002f,+8.497805198426792e-001f,+6.196027790386974e-002f,+3.441887418281308e-003f,
+1.984114073694576e-003f,+7.446643947564230e-002f,+8.511971524934298e-001f,+6.394448059633222e-002f,+3.273961348469476e-003f,
+2.146158355573756e-003f,+7.364060057620300e-002f,+8.523047035214708e-001f,+6.576906686507557e-002f,+3.112542065248617e-003f,
+2.301725477463282e-003f,+7.267746427298773e-002f,+8.531020949701688e-001f,+6.745250215166189e-002f,+2.946944771654501e-003f,
+2.462561691257243e-003f,+7.158263647902899e-002f,+8.535720573910749e-001f,+6.896640131950568e-002f,+2.787046434648109e-003f,
+2.620175869023839e-003f,+7.035330735092815e-002f,+8.537385600593657e-001f,+7.035330735092815e-002f,+2.620175869023839e-003f,
-1.327182200350553e-002f,-3.611589903135513e-001f,+3.611589903135513e-001f,+1.327182200350553e-002f,+0.000000000000000e+000f
};

/*
  \name QMF-Analysis-Synthesis-Twiddle

  \brief QMF analysis and synthesis twiddle factors
*/
//@{
/*   cos( PI / L * (n + 0.25) ) L=4 */
const float sbr_cos_twiddle_L04[] =
{
  0.9807852506637573f, 0.5555702447891235f
};

/*   cos( PI / L * (n + 0.25) ) L=8 */
const float sbr_cos_twiddle_L08[] =
{
  0.9951847791671753f, 0.8819212913513184f, 0.6343933343887329f, 0.2902846336364746f
};

/*   cos( PI / L * (n + 0.25) ) L=16 */
const float sbr_cos_twiddle_L16[] =
{
  0.9987955093383789f, 0.9700312614440918f, 0.9039893150329590f, 0.8032075166702271f,
  0.6715589761734009f, 0.5141026973724365f, 0.3368898630142212f, 0.1467304229736328f
};

/*   cos( PI / L * (n + 0.25) ) L=32 */
const float sbr_cos_twiddle_L32[] =
{
  0.9996988773345947f, 0.9924795627593994f, 0.9757021665573120f, 0.9495282173156738f,
  0.9142097234725952f, 0.8700870275497437f, 0.8175848722457886f, 0.7572088241577148f,
  0.6895405054092407f, 0.6152316331863403f, 0.5349975824356079f, 0.4496113061904907f,
  0.3598949909210205f, 0.2667127847671509f, 0.1709618568420410f, 0.0735645294189453f
};

/*   sin( PI / L * (n + 0.25) ) L=4 */
const float sbr_sin_twiddle_L04[] =
{
  0.1950902938842773f, 0.8314696550369263f
};

/*   sin( PI / L * (n + 0.25) ) L=8 */
const float sbr_sin_twiddle_L08[] =
{
  0.0980170965194702f, 0.4713966846466064f, 0.7730104923248291f, 0.9569402933120728f
};

/*   sin( PI / L * (n + 0.25) ) L=16 */
const float sbr_sin_twiddle_L16[] =
{
  0.0490676164627075f, 0.2429801225662231f, 0.4275550842285156f, 0.5956993103027344f,
  0.7409511804580688f, 0.8577286005020142f, 0.9415440559387207f, 0.9891765117645264f
};

/*  sin( PI / L * (n + 0.25) ) L=32 */
const float sbr_sin_twiddle_L32[] =
{
  0.0245412588119507f, 0.1224106550216675f, 0.2191011905670166f, 0.3136817216873169f,
  0.4052413702011108f, 0.4928982257843018f, 0.5758081674575806f, 0.6531728506088257f,
  0.7242470979690552f, 0.7883464097976685f, 0.8448535203933716f, 0.8932243585586548f,
  0.9329928159713745f, 0.9637761116027832f, 0.9852776527404785f, 0.9972904920578003f
};

/* sin( PI / L * n ) L=4 */
const float sbr_alt_sin_twiddle_L04[] =
{
  0.0000000000000000f, 0.7071068286895752f, 0.9999998807907104f
};

/* sin( PI / L * n ) L=8 */
const float sbr_alt_sin_twiddle_L08[] =
{
  0.0000000000000000f, 0.3826833963394165f, 0.7071068286895752f, 0.9238795042037964f,
  0.9999998807907104f
};

/* sin( PI / L * n ) L=16 */
const float sbr_alt_sin_twiddle_L16[] =
{
  0.0000000000000000f, 0.1950902938842773f, 0.3826833963394165f, 0.5555702447891235f,
  0.7071068286895752f, 0.8314696550369263f, 0.9238795042037964f, 0.9807852506637573f,
  0.9999998807907104f
};

/* sin( PI / L * n ) L=32 */
const float sbr_alt_sin_twiddle_L32[] =
{
  0.0000000000000000f, 0.0980170965194702f, 0.1950902938842773f, 0.2902846336364746f,
  0.3826833963394165f, 0.4713966846466064f, 0.5555702447891235f, 0.6343933343887329f,
  0.7071068286895752f, 0.7730104923248291f, 0.8314696550369263f, 0.8819212913513184f,
  0.9238795042037964f, 0.9569402933120728f, 0.9807852506637573f, 0.9951847791671753f,
  0.9999998807907104f
};

/* table of twiddle factors */
struct dct4Twiddle dct4TwiddleTable[] =
{
  {sbr_cos_twiddle_L04, sbr_sin_twiddle_L04, sbr_alt_sin_twiddle_L04},
  {sbr_cos_twiddle_L08, sbr_sin_twiddle_L08, sbr_alt_sin_twiddle_L08},
  {sbr_cos_twiddle_L16, sbr_sin_twiddle_L16, sbr_alt_sin_twiddle_L16},
  {sbr_cos_twiddle_L32, sbr_sin_twiddle_L32, sbr_alt_sin_twiddle_L32}
};
//@}

#ifndef LP_SBR_ONLY

//@}

/*
  \name QMF-Synthesis-Twiddle

  \brief QMF synthesis twiddle factors

  L=64, gain=1.0, angle = 1.0
*/
//@{
/*  cos (PI / L * (n + 0.25)); */
const float sbr_cos_twiddle_L64[] =
{
  0.99992470183914f,0.99811811290015f,0.99390697000236f,0.98730141815786f,
  0.97831737071963f,0.96697647104485f,0.95330604035419f,0.93733901191257f,
  0.91911385169006f,0.89867446569395f,0.87607009419541f,0.85135519310527f,
  0.82458930278503f,0.79583690460888f,0.76516726562246f,0.73265427167241f,
  0.69837624940897f,0.66241577759017f,0.62485948814239f,0.58579785745644f,
  0.54532498842205f,0.50353838372572f,0.46053871095824f,0.41642956009764f,
  0.37131719395184f,0.32531029216226f,0.27851968938505f,0.23105810828067f,
  0.18303988795514f,0.13458070850713f,0.08579731234444f,0.03680722294136f
};

/*  sin (PI / L * (n + 0.25)); */
const float sbr_sin_twiddle_L64[] =
{
  0.01227153828572f,0.06132073630221f,0.11022220729388f,0.15885814333386f,
  0.20711137619222f,0.25486565960451f,0.30200594931923f,0.34841868024943f,
  0.39399204006105f,0.43861623853853f,0.48218377207912f,0.52458968267847f,
  0.56573181078361f,0.60551104140433f,0.64383154288979f,0.68060099779545f,
  0.71573082528382f,0.74913639452346f,0.78073722857209f,0.81045719825259f,
  0.83822470555484f,0.86397285612159f,0.88763962040285f,0.90916798309052f,
  0.92850608047322f,0.94560732538052f,0.96043051941557f,0.97293995220556f,
  0.98310548743122f,0.99090263542778f,0.99631261218278f,0.99932238458835f,
};

/*  sin (PI / L * n) */
const float sbr_alt_sin_twiddle_L64[] =
{
  0.00000000000000f,0.04906767432742f,0.09801714032956f,0.14673047445536f,
  0.19509032201613f,0.24298017990326f,0.29028467725446f,0.33688985339222f,
  0.38268343236509f,0.42755509343028f,0.47139673682600f,0.51410274419322f,
  0.55557023301960f,0.59569930449243f,0.63439328416365f,0.67155895484702f,
  0.70710678118655f,0.74095112535496f,0.77301045336274f,0.80320753148064f,
  0.83146961230255f,0.85772861000027f,0.88192126434835f,0.90398929312344f,
  0.92387953251129f,0.94154406518302f,0.95694033573221f,0.97003125319454f,
  0.98078528040323f,0.98917650996478f,0.99518472667220f,0.99879545620517f,
  1.00000000000000f
};

const float sbr_cos_twiddle_ds_L32[] =
{
  0.99992470183914f, 0.99932238458835f, 0.99811811290015f, 0.99631261218278f,
  0.99390697000236f, 0.99090263542778f, 0.98730141815786f, 0.98310548743122f,
  0.97831737071963f, 0.97293995220556f, 0.96697647104485f, 0.96043051941557f,
  0.95330604035419f, 0.94560732538052f, 0.93733901191257f, 0.92850608047322f,
  0.91911385169006f, 0.90916798309052f, 0.89867446569395f, 0.88763962040285f,
  0.87607009419541f, 0.86397285612159f, 0.85135519310527f, 0.83822470555484f,
  0.82458930278503f, 0.81045719825259f, 0.79583690460888f, 0.78073722857209f,
  0.76516726562246f, 0.74913639452346f, 0.73265427167241f, 0.71573082528382f
};
const float sbr_sin_twiddle_ds_L32[] =
{
  0.01227153828572f, 0.03680722294136f, 0.06132073630221f, 0.08579731234444f,
  0.11022220729388f, 0.13458070850713f, 0.15885814333386f, 0.18303988795514f,
  0.20711137619222f, 0.23105810828067f, 0.25486565960451f, 0.27851968938505f,
  0.30200594931923f, 0.32531029216226f, 0.34841868024943f, 0.37131719395184f,
  0.39399204006105f, 0.41642956009764f, 0.43861623853853f, 0.46053871095824f,
  0.48218377207912f, 0.50353838372572f, 0.52458968267847f, 0.54532498842205f,
  0.56573181078361f, 0.58579785745644f, 0.60551104140433f, 0.62485948814239f,
  0.64383154288979f, 0.66241577759017f, 0.68060099779545f, 0.69837624940897f
};



//@}
//@{
/*
  \name QMF-Analysis-Twiddle

  \brief QMF analysis twiddle factors

  L=32, gain=2.0, angle = 0.75
*/
const float sbr_t_cos_L32[] =
{
  0.99932238458835f,0.99390697000236f,0.98310548743122f,0.96697647104485f,
  0.94560732538052f,0.91911385169006f,0.88763962040285f,0.85135519310527f,
  0.81045719825259f,0.76516726562246f,0.71573082528382f,0.66241577759017f,
  0.60551104140433f,0.54532498842205f,0.48218377207912f,0.41642956009764f,
  0.34841868024943f,0.27851968938505f,0.20711137619222f,0.13458070850713f,
  0.06132073630221f,-0.01227153828572f,-0.08579731234444f,-0.15885814333386f,
  -0.23105810828067f,-0.30200594931923f,-0.37131719395184f,-0.43861623853853f,
  -0.50353838372572f,-0.56573181078361f,-0.62485948814239f,-0.68060099779545f
};

const float sbr_t_sin_L32[] =
{
  0.03680722294136f,0.11022220729388f,0.18303988795514f,0.25486565960451f,
  0.32531029216226f,0.39399204006105f,0.46053871095824f,0.52458968267847f,
  0.58579785745644f,0.64383154288979f,0.69837624940897f,0.74913639452346f,
  0.79583690460888f,0.83822470555484f,0.87607009419541f,0.90916798309052f,
  0.93733901191257f,0.96043051941557f,0.97831737071963f,0.99090263542778f,
  0.99811811290015f,0.99992470183914f,0.99631261218278f,0.98730141815786f,
  0.97293995220556f,0.95330604035419f,0.92850608047322f,0.89867446569395f,
  0.86397285612159f,0.82458930278503f,0.78073722857209f,0.73265427167241f
};
#endif /* #ifndef LP_SBR_ONLY */

//@}


/*
  \name FrameInfoDefaults

  Predefined envelope positions for the FIX-FIX case (static framing)
*/
//@{
const FRAME_INFO sbr_frame_info1_16 = { 0, 1, {0, 16}, {1}, -1, 1, {0, 16} };
const FRAME_INFO sbr_frame_info2_16 = { 0, 2, {0, 8, 16}, {1, 1}, -1, 2, {0, 8, 16} };
const FRAME_INFO sbr_frame_info4_16 = { 0, 4, {0, 4, 8, 12, 16}, {1, 1, 1, 1}, -1, 2, {0, 8, 16} };


//@}

/*
  Default values for the time before the first SBR header is read
*/
const SBR_HEADER_DATA sbr_defaultHeader =
{
  SBR_NOT_INITIALIZED,
  0,
  0,
  16,
  2,
  1024,
  44100,


  UNDEFINED_CHANNEL_MODE,
  SBR_AMP_RES_3_0,


  5,
  0,
  0,
  SBR_FREQ_SCALE_DEFAULT,
  SBR_ALTER_SCALE_DEFAULT,
  SBR_NOISE_BANDS_DEFAULT,


  SBR_LIMITER_BANDS_DEFAULT,
  SBR_LIMITER_GAINS_DEFAULT,
  SBR_INTERPOL_FREQ_DEFAULT,
  SBR_SMOOTHING_LENGTH_DEFAULT,

  NULL

};


/*
  \name SBR_HuffmanTables

  SBR Huffman Table Overview:        \n
                                     \n
 o envelope level,   1.5 dB:         \n
    1)  sbr_huffBook_EnvLevel10T[120][2]   \n
    2)  sbr_huffBook_EnvLevel10F[120][2]   \n
                                     \n
 o envelope balance, 1.5 dB:         \n
    3)  sbr_huffBook_EnvBalance10T[48][2]  \n
    4)  sbr_huffBook_EnvBalance10F[48][2]  \n
                                     \n
 o envelope level,   3.0 dB:         \n
    5)  sbr_huffBook_EnvLevel11T[62][2]    \n
    6)  sbr_huffBook_EnvLevel11F[62][2]    \n
                                     \n
 o envelope balance, 3.0 dB:         \n
    7)  sbr_huffBook_EnvBalance11T[24][2]  \n
    8)  sbr_huffBook_EnvBalance11F[24][2]  \n
                                     \n
 o noise level,      3.0 dB:         \n
    9)  sbr_huffBook_NoiseLevel11T[62][2]  \n
    -) (sbr_huffBook_EnvLevel11F[62][2] is used for freq dir)\n
                                     \n
 o noise balance,    3.0 dB:         \n
   10)  sbr_huffBook_NoiseBalance11T[24][2]\n
    -) (sbr_huffBook_EnvBalance11F[24][2] is used for freq dir)\n
                                     \n
  (1.5 dB is never used for noise)

*/

const char sbr_huffBook_EnvLevel10T[120][2] = {
  {   1,   2 },    { -64, -65 },    {   3,   4 },    { -63, -66 },
  {   5,   6 },    { -62, -67 },    {   7,   8 },    { -61, -68 },
  {   9,  10 },    { -60, -69 },    {  11,  12 },    { -59, -70 },
  {  13,  14 },    { -58, -71 },    {  15,  16 },    { -57, -72 },
  {  17,  18 },    { -73, -56 },    {  19,  21 },    { -74,  20 },
  { -55, -75 },    {  22,  26 },    {  23,  24 },    { -54, -76 },
  { -77,  25 },    { -53, -78 },    {  27,  34 },    {  28,  29 },
  { -52, -79 },    {  30,  31 },    { -80, -51 },    {  32,  33 },
  { -83, -82 },    { -81, -50 },    {  35,  57 },    {  36,  40 },
  {  37,  38 },    { -88, -84 },    { -48,  39 },    { -90, -85 },
  {  41,  46 },    {  42,  43 },    { -49, -87 },    {  44,  45 },
  { -89, -86 },    {-124,-123 },    {  47,  50 },    {  48,  49 },
  {-122,-121 },    {-120,-119 },    {  51,  54 },    {  52,  53 },
  {-118,-117 },    {-116,-115 },    {  55,  56 },    {-114,-113 },
  {-112,-111 },    {  58,  89 },    {  59,  74 },    {  60,  67 },
  {  61,  64 },    {  62,  63 },    {-110,-109 },    {-108,-107 },
  {  65,  66 },    {-106,-105 },    {-104,-103 },    {  68,  71 },
  {  69,  70 },    {-102,-101 },    {-100, -99 },    {  72,  73 },
  { -98, -97 },    { -96, -95 },    {  75,  82 },    {  76,  79 },
  {  77,  78 },    { -94, -93 },    { -92, -91 },    {  80,  81 },
  { -47, -46 },    { -45, -44 },    {  83,  86 },    {  84,  85 },
  { -43, -42 },    { -41, -40 },    {  87,  88 },    { -39, -38 },
  { -37, -36 },    {  90, 105 },    {  91,  98 },    {  92,  95 },
  {  93,  94 },    { -35, -34 },    { -33, -32 },    {  96,  97 },
  { -31, -30 },    { -29, -28 },    {  99, 102 },    { 100, 101 },
  { -27, -26 },    { -25, -24 },    { 103, 104 },    { -23, -22 },
  { -21, -20 },    { 106, 113 },    { 107, 110 },    { 108, 109 },
  { -19, -18 },    { -17, -16 },    { 111, 112 },    { -15, -14 },
  { -13, -12 },    { 114, 117 },    { 115, 116 },    { -11, -10 },
  {  -9,  -8 },    { 118, 119 },    {  -7,  -6 },    {  -5,  -4 }
};

const char sbr_huffBook_EnvLevel10F[120][2] = {
  {   1,   2 },    { -64, -65 },    {   3,   4 },    { -63, -66 },
  {   5,   6 },    { -67, -62 },    {   7,   8 },    { -68, -61 },
  {   9,  10 },    { -69, -60 },    {  11,  13 },    { -70,  12 },
  { -59, -71 },    {  14,  16 },    { -58,  15 },    { -72, -57 },
  {  17,  19 },    { -73,  18 },    { -56, -74 },    {  20,  23 },
  {  21,  22 },    { -55, -75 },    { -54, -53 },    {  24,  27 },
  {  25,  26 },    { -76, -52 },    { -77, -51 },    {  28,  31 },
  {  29,  30 },    { -50, -78 },    { -79, -49 },    {  32,  36 },
  {  33,  34 },    { -48, -47 },    { -80,  35 },    { -81, -82 },
  {  37,  47 },    {  38,  41 },    {  39,  40 },    { -83, -46 },
  { -45, -84 },    {  42,  44 },    { -85,  43 },    { -44, -43 },
  {  45,  46 },    { -88, -87 },    { -86, -90 },    {  48,  66 },
  {  49,  56 },    {  50,  53 },    {  51,  52 },    { -92, -42 },
  { -41, -39 },    {  54,  55 },    {-105, -89 },    { -38, -37 },
  {  57,  60 },    {  58,  59 },    { -94, -91 },    { -40, -36 },
  {  61,  63 },    { -20,  62 },    {-115,-110 },    {  64,  65 },
  {-108,-107 },    {-101, -97 },    {  67,  89 },    {  68,  75 },
  {  69,  72 },    {  70,  71 },    { -95, -93 },    { -34, -27 },
  {  73,  74 },    { -22, -17 },    { -16,-124 },    {  76,  82 },
  {  77,  79 },    {-123,  78 },    {-122,-121 },    {  80,  81 },
  {-120,-119 },    {-118,-117 },    {  83,  86 },    {  84,  85 },
  {-116,-114 },    {-113,-112 },    {  87,  88 },    {-111,-109 },
  {-106,-104 },    {  90, 105 },    {  91,  98 },    {  92,  95 },
  {  93,  94 },    {-103,-102 },    {-100, -99 },    {  96,  97 },
  { -98, -96 },    { -35, -33 },    {  99, 102 },    { 100, 101 },
  { -32, -31 },    { -30, -29 },    { 103, 104 },    { -28, -26 },
  { -25, -24 },    { 106, 113 },    { 107, 110 },    { 108, 109 },
  { -23, -21 },    { -19, -18 },    { 111, 112 },    { -15, -14 },
  { -13, -12 },    { 114, 117 },    { 115, 116 },    { -11, -10 },
  {  -9,  -8 },    { 118, 119 },    {  -7,  -6 },    {  -5,  -4 }
};

const char sbr_huffBook_EnvBalance10T[48][2] = {
  { -64,   1 },    { -63,   2 },    { -65,   3 },    { -62,   4 },
  { -66,   5 },    { -61,   6 },    { -67,   7 },    { -60,   8 },
  { -68,   9 },    {  10,  11 },    { -69, -59 },    {  12,  13 },
  { -70, -58 },    {  14,  28 },    {  15,  21 },    {  16,  18 },
  { -57,  17 },    { -71, -56 },    {  19,  20 },    { -88, -87 },
  { -86, -85 },    {  22,  25 },    {  23,  24 },    { -84, -83 },
  { -82, -81 },    {  26,  27 },    { -80, -79 },    { -78, -77 },
  {  29,  36 },    {  30,  33 },    {  31,  32 },    { -76, -75 },
  { -74, -73 },    {  34,  35 },    { -72, -55 },    { -54, -53 },
  {  37,  41 },    {  38,  39 },    { -52, -51 },    { -50,  40 },
  { -49, -48 },    {  42,  45 },    {  43,  44 },    { -47, -46 },
  { -45, -44 },    {  46,  47 },    { -43, -42 },    { -41, -40 }
};

const char sbr_huffBook_EnvBalance10F[48][2] = {
  { -64,   1 },    { -65,   2 },    { -63,   3 },    { -66,   4 },
  { -62,   5 },    { -61,   6 },    { -67,   7 },    { -68,   8 },
  { -60,   9 },    {  10,  11 },    { -69, -59 },    { -70,  12 },
  { -58,  13 },    {  14,  17 },    { -71,  15 },    { -57,  16 },
  { -56, -73 },    {  18,  32 },    {  19,  25 },    {  20,  22 },
  { -72,  21 },    { -88, -87 },    {  23,  24 },    { -86, -85 },
  { -84, -83 },    {  26,  29 },    {  27,  28 },    { -82, -81 },
  { -80, -79 },    {  30,  31 },    { -78, -77 },    { -76, -75 },
  {  33,  40 },    {  34,  37 },    {  35,  36 },    { -74, -55 },
  { -54, -53 },    {  38,  39 },    { -52, -51 },    { -50, -49 },
  {  41,  44 },    {  42,  43 },    { -48, -47 },    { -46, -45 },
  {  45,  46 },    { -44, -43 },    { -42,  47 },    { -41, -40 }
};

const char sbr_huffBook_EnvLevel11T[62][2] = {
  { -64,   1 },    { -65,   2 },    { -63,   3 },    { -66,   4 },
  { -62,   5 },    { -67,   6 },    { -61,   7 },    { -68,   8 },
  { -60,   9 },    {  10,  11 },    { -69, -59 },    {  12,  14 },
  { -70,  13 },    { -71, -58 },    {  15,  18 },    {  16,  17 },
  { -72, -57 },    { -73, -74 },    {  19,  22 },    { -56,  20 },
  { -55,  21 },    { -54, -77 },    {  23,  31 },    {  24,  25 },
  { -75, -76 },    {  26,  27 },    { -78, -53 },    {  28,  29 },
  { -52, -95 },    { -94,  30 },    { -93, -92 },    {  32,  47 },
  {  33,  40 },    {  34,  37 },    {  35,  36 },    { -91, -90 },
  { -89, -88 },    {  38,  39 },    { -87, -86 },    { -85, -84 },
  {  41,  44 },    {  42,  43 },    { -83, -82 },    { -81, -80 },
  {  45,  46 },    { -79, -51 },    { -50, -49 },    {  48,  55 },
  {  49,  52 },    {  50,  51 },    { -48, -47 },    { -46, -45 },
  {  53,  54 },    { -44, -43 },    { -42, -41 },    {  56,  59 },
  {  57,  58 },    { -40, -39 },    { -38, -37 },    {  60,  61 },
  { -36, -35 },    { -34, -33 }
};

const char sbr_huffBook_EnvLevel11F[62][2] = {
  { -64,   1 },    { -65,   2 },    { -63,   3 },    { -66,   4 },
  { -62,   5 },    { -67,   6 },    {   7,   8 },    { -61, -68 },
  {   9,  10 },    { -60, -69 },    {  11,  12 },    { -59, -70 },
  {  13,  14 },    { -58, -71 },    {  15,  16 },    { -57, -72 },
  {  17,  19 },    { -56,  18 },    { -55, -73 },    {  20,  24 },
  {  21,  22 },    { -74, -54 },    { -53,  23 },    { -75, -76 },
  {  25,  30 },    {  26,  27 },    { -52, -51 },    {  28,  29 },
  { -77, -79 },    { -50, -49 },    {  31,  39 },    {  32,  35 },
  {  33,  34 },    { -78, -46 },    { -82, -88 },    {  36,  37 },
  { -83, -48 },    { -47,  38 },    { -86, -85 },    {  40,  47 },
  {  41,  44 },    {  42,  43 },    { -80, -44 },    { -43, -42 },
  {  45,  46 },    { -39, -87 },    { -84, -40 },    {  48,  55 },
  {  49,  52 },    {  50,  51 },    { -95, -94 },    { -93, -92 },
  {  53,  54 },    { -91, -90 },    { -89, -81 },    {  56,  59 },
  {  57,  58 },    { -45, -41 },    { -38, -37 },    {  60,  61 },
  { -36, -35 },    { -34, -33 }
};

const char sbr_huffBook_EnvBalance11T[24][2] = {
  { -64,   1 },    { -63,   2 },    { -65,   3 },    { -66,   4 },
  { -62,   5 },    { -61,   6 },    { -67,   7 },    { -68,   8 },
  { -60,   9 },    {  10,  16 },    {  11,  13 },    { -69,  12 },
  { -76, -75 },    {  14,  15 },    { -74, -73 },    { -72, -71 },
  {  17,  20 },    {  18,  19 },    { -70, -59 },    { -58, -57 },
  {  21,  22 },    { -56, -55 },    { -54,  23 },    { -53, -52 }
};

const char sbr_huffBook_EnvBalance11F[24][2] = {
  { -64,   1 },    { -65,   2 },    { -63,   3 },    { -66,   4 },
  { -62,   5 },    { -61,   6 },    { -67,   7 },    { -68,   8 },
  { -60,   9 },    {  10,  13 },    { -69,  11 },    { -59,  12 },
  { -58, -76 },    {  14,  17 },    {  15,  16 },    { -75, -74 },
  { -73, -72 },    {  18,  21 },    {  19,  20 },    { -71, -70 },
  { -57, -56 },    {  22,  23 },    { -55, -54 },    { -53, -52 }
};

const char sbr_huffBook_NoiseLevel11T[62][2] = {
  { -64,   1 },    { -63,   2 },    { -65,   3 },    { -66,   4 },
  { -62,   5 },    { -67,   6 },    {   7,   8 },    { -61, -68 },
  {   9,  30 },    {  10,  15 },    { -60,  11 },    { -69,  12 },
  {  13,  14 },    { -59, -53 },    { -95, -94 },    {  16,  23 },
  {  17,  20 },    {  18,  19 },    { -93, -92 },    { -91, -90 },
  {  21,  22 },    { -89, -88 },    { -87, -86 },    {  24,  27 },
  {  25,  26 },    { -85, -84 },    { -83, -82 },    {  28,  29 },
  { -81, -80 },    { -79, -78 },    {  31,  46 },    {  32,  39 },
  {  33,  36 },    {  34,  35 },    { -77, -76 },    { -75, -74 },
  {  37,  38 },    { -73, -72 },    { -71, -70 },    {  40,  43 },
  {  41,  42 },    { -58, -57 },    { -56, -55 },    {  44,  45 },
  { -54, -52 },    { -51, -50 },    {  47,  54 },    {  48,  51 },
  {  49,  50 },    { -49, -48 },    { -47, -46 },    {  52,  53 },
  { -45, -44 },    { -43, -42 },    {  55,  58 },    {  56,  57 },
  { -41, -40 },    { -39, -38 },    {  59,  60 },    { -37, -36 },
  { -35,  61 },    { -34, -33 }
};

const char sbr_huffBook_NoiseBalance11T[24][2] = {
  { -64,   1 },    { -65,   2 },    { -63,   3 },    {   4,   9 },
  { -66,   5 },    { -62,   6 },    {   7,   8 },    { -76, -75 },
  { -74, -73 },    {  10,  17 },    {  11,  14 },    {  12,  13 },
  { -72, -71 },    { -70, -69 },    {  15,  16 },    { -68, -67 },
  { -61, -60 },    {  18,  21 },    {  19,  20 },    { -59, -58 },
  { -57, -56 },    {  22,  23 },    { -55, -54 },    { -53, -52 }
};
//@}



const float aRevLinkDecaySer[NO_SERIAL_ALLPASS_LINKS]= {
  6.51439057531056e-001f,
  5.64718122007759e-001f,
  4.89541659556953e-001f
};

const char  aRevLinkDelaySer[] = {3,  4,  5};

const char groupBorders[NO_IID_GROUPS + 1] =
{
     4,  5,  0,  1,  2,  3,
     7,  6,
     8,  9,
     3,  4,  5,  6,  7,  8,  9,  11, 14, 18, 23, 35, 64
};

const short bins2groupMap[NO_IID_GROUPS] =
{
    ( NEGATE_IPD_MASK | 1 ), ( NEGATE_IPD_MASK | 0 ),
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19
};


const float aFractDelayPhaseFactorReQmf[NO_QMF_ALLPASS_CHANNELS]=
{
  -0.411514f,
  0.718126f,  0.898028f,  -0.109734f,  -0.972370f,
  -0.549023f,  0.600420f,  0.955793f,  0.047107f,
  -0.923879f,  -0.673013f,  0.467929f,  0.990024f,
  0.202788f,  -0.852640f,  -0.780431f,  0.323917f,
  0.999877f,  0.353476f,  -0.760405f
};

const float aFractDelayPhaseFactorImQmf[NO_QMF_ALLPASS_CHANNELS]=
{
  0.911403f,
  0.695913f,  -0.439939f,  -0.993961f,  -0.233446f,
  0.835807f,  0.799685f,  -0.294040f,  -0.998890f,
  -0.382684f,  0.739631f,  0.883766f,  -0.140901f,
  -0.979223f,  -0.522499f,  0.625242f,  0.946086f,
  0.015708f,  -0.935444f,  -0.649449f
};

const float aFractDelayPhaseFactorReSubQmf[SUBQMF_GROUPS]=
{
   0.988295f,   0.896293f,   0.720854f,   0.478309f,
   0.896293f,   0.988295f,
  -0.542441f,   0.039260f,  -0.926857f,  -0.974173f
};

const float aFractDelayPhaseFactorImSubQmf[SUBQMF_GROUPS]=
{
  -0.152555f,  -0.443462f,  -0.693087f,  -0.878192f,
   0.443462f,   0.152555f,
  -0.840094f,  -0.999229f,  -0.375416f,   0.225801f
};

const float aFractDelayPhaseFactorSerReQmf0[NO_QMF_ALLPASS_CHANNELS]=
{
  0.015707f,
  0.979223f,  0.411514f,  -0.799685f,  -0.760406f,
  0.467930f,  0.964557f,  -0.047107f,  -0.985109f,
  -0.382683f,  0.818150f,  0.739631f,  -0.495459f,
  -0.955793f,  0.078459f,  0.990024f,  0.353474f,
  -0.835808f,  -0.718126f,  0.522499f
};

const float aFractDelayPhaseFactorSerReQmf1[NO_QMF_ALLPASS_CHANNELS]=
{
  -0.382683f,
  -0.382683f,  0.923880f,  -0.923880f,  0.382683f,
  0.382683f,  -0.923880f,  0.923880f,  -0.382683f,
  -0.382683f,  0.923880f,  -0.923880f,  0.382683f,
  0.382683f,  -0.923880f,  0.923880f,  -0.382683f,
  -0.382683f,  0.923880f,  -0.923880f
};

const float aFractDelayPhaseFactorSerReQmf2[NO_QMF_ALLPASS_CHANNELS]=
{
  -0.781412f,
  0.192008f,  0.958968f,  0.694784f,  -0.316477f,
  -0.987441f,  -0.596645f,  0.435703f,  0.999555f,
  0.488621f,  -0.547709f,  -0.995107f,  -0.372502f,
  0.650642f,  0.974173f,  0.250211f,  -0.742795f,
  -0.937099f,  -0.123774f,  0.822641f
};

const float aFractDelayPhaseFactorSerImQmf0[NO_QMF_ALLPASS_CHANNELS]=
{
  0.999877f,
  0.202787f,  -0.911403f,  -0.600420f,  0.649448f,
  0.883766f,  -0.263873f,  -0.998890f,  -0.171929f,
  0.923880f,  0.575005f,  -0.673013f,  -0.868631f,
  0.294041f,  0.996917f,  0.140901f,  -0.935444f,
  -0.549022f,  0.695913f,  0.852640f
};

const float aFractDelayPhaseFactorSerImQmf1[NO_QMF_ALLPASS_CHANNELS]=
{
  -0.923880f,
  0.923880f,  -0.382683f,  -0.382683f,  0.923880f,
  -0.923880f,  0.382683f,  0.382683f,  -0.923880f,
  0.923880f,  -0.382683f,  -0.382683f,  0.923880f,
  -0.923880f,  0.382683f,  0.382683f,  -0.923880f,
  0.923880f,  -0.382683f,  -0.382683f
};

const float aFractDelayPhaseFactorSerImQmf2[NO_QMF_ALLPASS_CHANNELS]=
{
  0.624016f,
  0.981393f,  0.283513f,  -0.719219f,  -0.948600f,
  -0.157986f,  0.802505f,  0.900091f,  0.029841f,
  -0.872496f,  -0.836669f,  0.098799f,  0.928032f,
  0.759385f,  -0.225801f,  -0.968191f,  -0.669519f,
  0.349063f,  0.992310f,  0.568562f
};

const float aFractDelayPhaseFactorSerReSubQmf0[SUBQMF_GROUPS]=
{
   0.985777f,   0.874408f,   0.664252f,   0.379052f,
   0.874408f,   0.985777f,
  -0.712639f,  -0.117537f,  -0.994792f,  -0.840093f
};

const float aFractDelayPhaseFactorSerReSubQmf1[SUBQMF_GROUPS]=
{
   0.956940f,   0.634393f,   0.098017f,  -0.471397f,
   0.634393f,   0.956940f,
  -0.555570f,  -0.980785f,   0.555570f,   0.980785f
};
const float aFractDelayPhaseFactorSerReSubQmf2[SUBQMF_GROUPS]=
{
   0.990730f,   0.917599f,   0.776734f,   0.578534f,
   0.917599f,   0.990730f,
  -0.330597f,   0.206631f,  -0.772013f,  -0.989689f
};

const float aFractDelayPhaseFactorSerImSubQmf0[SUBQMF_GROUPS]=
{
  -0.168059f,  -0.485191f,  -0.747508f,  -0.925375f,
   0.485191f,   0.168059f,
  -0.701531f,  -0.993068f,  -0.101924f,   0.542442f
};

const float aFractDelayPhaseFactorSerImSubQmf1[SUBQMF_GROUPS]=
{
  -0.290285f,  -0.773010f,  -0.995185f,  -0.881921f,
   0.773010f,   0.290285f,
   0.831470f,  -0.195090f,   0.831470f,  -0.195090f
};

const float aFractDelayPhaseFactorSerImSubQmf2[SUBQMF_GROUPS]=
{
  -0.135845f,  -0.397508f,  -0.629829f,  -0.815658f,
   0.397508f,   0.135845f,
  -0.943772f,  -0.978419f,  -0.635607f,  -0.143234f
};

const float *aaFractDelayPhaseFactorSerReQmf[3]=
{
  aFractDelayPhaseFactorSerReQmf0,
  aFractDelayPhaseFactorSerReQmf1,
  aFractDelayPhaseFactorSerReQmf2
};
const float *aaFractDelayPhaseFactorSerImQmf[3]=
{
  aFractDelayPhaseFactorSerImQmf0,
  aFractDelayPhaseFactorSerImQmf1,
  aFractDelayPhaseFactorSerImQmf2
};
const float *aaFractDelayPhaseFactorSerReSubQmf[3]=
{
  aFractDelayPhaseFactorSerReSubQmf0,
  aFractDelayPhaseFactorSerReSubQmf1,
  aFractDelayPhaseFactorSerReSubQmf2
};
const float *aaFractDelayPhaseFactorSerImSubQmf[3]=
{
  aFractDelayPhaseFactorSerImSubQmf0,
  aFractDelayPhaseFactorSerImSubQmf1,
  aFractDelayPhaseFactorSerImSubQmf2
};

const float scaleFactors[NO_IID_LEVELS]=
{
  1.411983f,  1.403138f,  1.386877f,  1.348400f,
  1.291249f,  1.196037f,  1.107372f,  1.000000f,
  0.879617f,  0.754649f,  0.576780f,  0.426401f,
  0.276718f,  0.176645f,  0.079402f
};

const float scaleFactorsFine[NO_IID_LEVELS_FINE]=
{
  1.414207f,  1.414191f,  1.414143f,  1.413990f,
  1.413507f,  1.411983f,  1.409773f,  1.405395f,
  1.396780f,  1.380053f,  1.348400f,  1.313920f,
  1.264310f,  1.196037f,  1.107372f,  1.000000f,
  0.879617f,  0.754649f,  0.633656f,  0.523081f,
  0.426401f,  0.308955f,  0.221375f,  0.157688f,
  0.111982f,  0.079402f,  0.044699f,  0.025145f,
  0.014141f,  0.007953f,  0.004472f
};

const float alphas[NO_ICC_LEVELS]=
{
  0.000000f,  0.178428f,  0.285667f,  0.463072f,
  0.597163f,  0.785398f,  1.100309f,  1.570796f
};

const float p2_6[6] =
{
   0.01899487526049f, -0.07293139167538f,
   0.30596630545168f,  0.30596630545168f,
  -0.07293139167538f,  0.01899487526049f
};

const float p8_13[13] =
{
  -0.00746082949812f, -0.02270420949825f, -0.04546865930473f, -0.07266113929591f,
  -0.09885108575264f,  0.11793710567217f,  0.125f,             0.11793710567217f,
   0.09885108575264f,  0.07266113929591f,  0.04546865930473f,  0.02270420949825f,
   0.00746082949812f
};

/* IID & ICC Huffman codebooks */
const char aBookPsIidTimeDecode[28][2] = {
  { -64,   1 },    { -65,   2 },    { -63,   3 },    { -66,   4 },
  { -62,   5 },    { -67,   6 },    { -61,   7 },    { -68,   8 },
  { -60,   9 },    { -69,  10 },    { -59,  11 },    { -70,  12 },
  { -58,  13 },    { -57,  14 },    { -71,  15 },    {  16,  17 },
  { -56, -72 },    {  18,  21 },    {  19,  20 },    { -55, -78 },
  { -77, -76 },    {  22,  25 },    {  23,  24 },    { -75, -74 },
  { -73, -54 },    {  26,  27 },    { -53, -52 },    { -51, -50 }
};

const char aBookPsIidFreqDecode[28][2] = {
  { -64,   1 },    {   2,   3 },    { -63, -65 },    {   4,   5 },
  { -62, -66 },    {   6,   7 },    { -61, -67 },    {   8,   9 },
  { -68, -60 },    { -59,  10 },    { -69,  11 },    { -58,  12 },
  { -70,  13 },    { -71,  14 },    { -57,  15 },    {  16,  17 },
  { -56, -72 },    {  18,  19 },    { -55, -54 },    {  20,  21 },
  { -73, -53 },    {  22,  24 },    { -74,  23 },    { -75, -78 },
  {  25,  26 },    { -77, -76 },    { -52,  27 },    { -51, -50 }
};

const char aBookPsIccTimeDecode[14][2] = {
  { -64,   1 },    { -63,   2 },    { -65,   3 },    { -62,   4 },
  { -66,   5 },    { -61,   6 },    { -67,   7 },    { -60,   8 },
  { -68,   9 },    { -59,  10 },    { -69,  11 },    { -58,  12 },
  { -70,  13 },    { -71, -57 }
};

const char aBookPsIccFreqDecode[14][2] = {
  { -64,   1 },    { -63,   2 },    { -65,   3 },    { -62,   4 },
  { -66,   5 },    { -61,   6 },    { -67,   7 },    { -60,   8 },
  { -59,   9 },    { -68,  10 },    { -58,  11 },    { -69,  12 },
  { -57,  13 },    { -70, -71 }
};

const char aBookPsIidFineTimeDecode[60][2] = {
  {   1, -64 },    { -63,   2 },    {   3, -65 },    {   4,  59 },
  {   5,   7 },    {   6, -67 },    { -68, -60 },    { -61,   8 },
  {   9,  11 },    { -59,  10 },    { -70, -58 },    {  12,  41 },
  {  13,  20 },    {  14, -71 },    { -55,  15 },    { -53,  16 },
  {  17, -77 },    {  18,  19 },    { -85, -84 },    { -46, -45 },
  { -57,  21 },    {  22,  40 },    {  23,  29 },    { -51,  24 },
  {  25,  26 },    { -83, -82 },    {  27,  28 },    { -90, -38 },
  { -92, -91 },    {  30,  37 },    {  31,  34 },    {  32,  33 },
  { -35, -34 },    { -37, -36 },    {  35,  36 },    { -94, -93 },
  { -89, -39 },    {  38, -79 },    {  39, -81 },    { -88, -40 },
  { -74, -54 },    {  42, -69 },    {  43,  44 },    { -72, -56 },
  {  45,  52 },    {  46,  50 },    {  47, -76 },    { -49,  48 },
  { -47,  49 },    { -87, -41 },    { -52,  51 },    { -78, -50 },
  {  53, -73 },    {  54, -75 },    {  55,  57 },    {  56, -80 },
  { -86, -42 },    { -48,  58 },    { -44, -43 },    { -66, -62 }
};

const char aBookPsIidFineFreqDecode[60][2] = {
  {   1, -64 },    {   2,   4 },    {   3, -65 },    { -66, -62 },
  { -63,   5 },    {   6,   7 },    { -67, -61 },    {   8,   9 },
  { -68, -60 },    {  10,  11 },    { -69, -59 },    {  12,  13 },
  { -70, -58 },    {  14,  18 },    { -57,  15 },    {  16, -72 },
  { -54,  17 },    { -75, -53 },    {  19,  37 },    { -56,  20 },
  {  21, -73 },    {  22,  29 },    {  23, -76 },    {  24, -78 },
  {  25,  28 },    {  26,  27 },    { -85, -43 },    { -83, -45 },
  { -81, -47 },    { -52,  30 },    { -50,  31 },    {  32, -79 },
  {  33,  34 },    { -82, -46 },    {  35,  36 },    { -90, -89 },
  { -92, -91 },    {  38, -71 },    { -55,  39 },    {  40, -74 },
  {  41,  50 },    {  42, -77 },    { -49,  43 },    {  44,  47 },
  {  45,  46 },    { -86, -42 },    { -88, -87 },    {  48,  49 },
  { -39, -38 },    { -41, -40 },    { -51,  51 },    {  52,  59 },
  {  53,  56 },    {  54,  55 },    { -35, -34 },    { -37, -36 },
  {  57,  58 },    { -94, -93 },    { -84, -44 },    { -80, -48 }
};

